<?php
namespace App\Services\Dashboard;

use DB;

class DashboardService
{
    public function getDashboardData($request)
    {   
        $matrade_achivement_data = null;
        $export_promotions_data = null;
        $exporters_development_data = null;
        $date_range = $this->getValuesFromFilter($request);
        
        $matrade_achivement_data = $this->getMatradeAchivementData($date_range);
        $export_promotions_data = $this->getExportPromotionsData($date_range);
        $exporters_development_data = $this->getExportersDevelopmentData($date_range);
        
        return (object)[
            'matrade_achievement' => $matrade_achivement_data,
            'export_promotions' => $export_promotions_data,
            'exporters_development' => $exporters_development_data,
            'date_range' => $date_range,
        ];
    }

    private function getValuesFromFilter($request)
    {   
        $periods = [
            'Q1' => ['start' => '01-01', 'end' => '03-31', 'quarter' => [1]],
            'Q2' => ['start' => '04-01', 'end' => '06-30', 'quarter' => [2]],
            'Q3' => ['start' => '07-01', 'end' => '09-30', 'quarter' => [3]],
            'Q4' => ['start' => '10-01', 'end' => '12-31', 'quarter' => [4]],
            'H1' => ['start' => '01-01', 'end' => '06-30', 'quarter' => [1,2]],
            'H2' => ['start' => '07-01', 'end' => '12-31', 'quarter' => [3,4]],
            'F'  => ['start' => '01-01', 'end' => '12-31', 'quarter' => [1,2,3,4]], // When No filter Choosen
        ];

        // When No Filter Choosen
        $year_from = (int) ($request->yearFrom ?: date('Y'));
        $year_to   = (int) ($request->yearTo   ?: $year_from);

        //****need to work */
        if ($year_to < $year_from) {
            //return "Year To must be greater than or equal to Year From.");
        }

        $duration_from = isset($periods[$request->durationFrom]) ? $request->durationFrom : 'F';
        $duration_to   = isset($periods[$request->durationTo])   ? $request->durationTo   : 'F';


        $start_date = "{$year_from}-{$periods[$duration_from]['start']}";
        $end_date   = "{$year_to}-{$periods[$duration_to]['end']}";
        

        return (object) [
            'start_date'    => $start_date,
            'end_date'      => $end_date,
            'year_from'     => $year_from,
            'duration_from' => $duration_from,
            'year_to'       => $year_to,
            'duration_to'   => $duration_to,
            'quarter_from'    => $periods[$duration_from]['quarter'],
            'quarter_to'      => $periods[$duration_to]['quarter'],
        ];
    }

    public function getMatradeAchivementData($date_range) //Only Year will Compute
    {
        $indicators = [
            'programme'         => 'Programme',
            'export_sales_rm'   => 'Export Sales',
            'company_assisted'  => 'Companies assisted'
        ];

        $matrade_achivement = [];

        foreach ($indicators as $key => $indicator) {
            $result = DB::table('kpi_indicator as ki')
                        ->join('kpi_achievement as ka', 'ki.kpi_achievement_id', '=', 'ka.kpi_achievement_id')
                        ->join('kpi_achievement_detail as kad', 'ka.kpi_achievement_det_id', '=', 'kad.kpi_achievement_det_id')
                        ->select([
                            DB::raw('SUM(ka.cumulative_achievement) as total_achievement'),
                            DB::raw('SUM(ka.annual_target) as total_target')
                        ])
                        ->where('ki.indicator', $indicator)
                        ->where(function ($query) use ($date_range) {
                            $query->whereBetween(
                                DB::raw("
                                    CASE 
                                        WHEN kad.start_date IS NOT NULL THEN kad.start_date
                                        WHEN kad.approved_date IS NOT NULL THEN kad.approved_date
                                        ELSE NULL
                                    END
                                "), [$date_range->start_date, $date_range->end_date]
                            )
                            ->orWhere(function ($q) use ($date_range) {
                                $q->whereNull('kad.start_date')
                                ->whereNull('kad.approved_date')
                                ->whereBetween('kad.year', [$date_range->year_from, $date_range->year_to])
                                ->whereIn('kad.quarter', array_unique(array_merge($date_range->quarter_from,
                                $date_range->quarter_to)));
                            });
                        })
                        ->first();

            $matrade_achivement[$key] = (object)[
                'total_achievement' => (int) $result->total_achievement,
                'total_target'      => (int) $result->total_target,
                'total_achievement_short' => $this->formatForShow($result->total_achievement),
                'total_target_short'      => $this->formatForShow($result->total_target),
                'percentage' => $this->calculateAchievementPercentage($result),
            ];
        }

        return (object)$matrade_achivement;
    }


    public function getExportPromotionsData($date_range)
    {   
        $indicators = [
            'programme'          => 'Programme',
            'export_sales_rm'    => 'Export Sales',
            'company_assisted'   => 'Companies Assisted'
        ];

        $export_promotion = [];

        foreach ($indicators as $key => $indicator) {
            $result = DB::table('kpi_indicator as ki')
                        ->join('kpi_achievement as ka', 'ki.kpi_achievement_id', '=', 'ka.kpi_achievement_id')
                        ->join('kpi_achievement_detail as kad', 'ka.kpi_achievement_det_id', '=', 'kad.kpi_achievement_det_id')
                        ->select([
                            DB::raw('SUM(ka.cumulative_achievement) as total_achievement'),
                            DB::raw('SUM(ka.annual_target) as total_target')
                        ])
                        ->where('ki.indicator', $indicator)
                        ->where('ki.Component_Achievement', 'promotion')
                        ->where(function ($query) use ($date_range) {
                            $query->whereBetween(
                                DB::raw("
                                    CASE 
                                        WHEN kad.start_date IS NOT NULL THEN kad.start_date
                                        WHEN kad.approved_date IS NOT NULL THEN kad.approved_date
                                        ELSE NULL
                                    END
                                "), [$date_range->start_date, $date_range->end_date]
                            )
                            ->orWhere(function ($q) use ($date_range) {
                                $q->whereNull('kad.start_date')
                                ->whereNull('kad.approved_date')
                                ->whereBetween('kad.year', [$date_range->year_from, $date_range->year_to])
                                ->whereIn('kad.quarter', array_unique(array_merge($date_range->quarter_from,
                                $date_range->quarter_to)));
                            });
                        })
                        ->first();

            $export_promotion[$key] = (object)[
                'total_achievement'       => (int) $result->total_achievement,
                'total_target'            => (int) $result->total_target,
                'total_achievement_short' => $this->formatForShow($result->total_achievement),
                'total_target_short'      => $this->formatForShow($result->total_target),
                'percentage' => $this->calculateAchievementPercentage($result),
            ];
        }
        return (object)$export_promotion;
    }

    public function getExportersDevelopmentData($date_range)
    {
        $indicators = [
            'programme'         => 'Programme',
            'export_sales_rm'   => 'Export Sales',
            'company_assisted'  => 'Companies Assisted'
        ];

        $exporters_development = [];

        foreach ($indicators as $key => $indicator) {
            $result = DB::table('kpi_indicator as ki')
                        ->join('kpi_achievement as ka', 'ki.kpi_achievement_id', '=', 'ka.kpi_achievement_id')
                        ->join('kpi_achievement_detail as kad', 'ka.kpi_achievement_det_id', '=', 'kad.kpi_achievement_det_id')
                        ->select([
                            DB::raw('SUM(ka.cumulative_achievement) as total_achievement'),
                            DB::raw('SUM(ka.annual_target) as total_target')
                        ])
                        ->where('ki.indicator', $indicator)
                        ->where('ki.Component_Achievement', 'development')
                        ->where(function ($query) use ($date_range) {
                            $query->whereBetween(
                                DB::raw("
                                    CASE 
                                        WHEN kad.start_date IS NOT NULL THEN kad.start_date
                                        WHEN kad.approved_date IS NOT NULL THEN kad.approved_date
                                        ELSE NULL
                                    END
                                "), [$date_range->start_date, $date_range->end_date]
                            )
                            ->orWhere(function ($q) use ($date_range) {
                                $q->whereNull('kad.start_date')
                                ->whereNull('kad.approved_date')
                                ->whereBetween('kad.year', [$date_range->year_from, $date_range->year_to])
                                ->whereIn('kad.quarter', array_unique(array_merge($date_range->quarter_from,
                                $date_range->quarter_to)));
                            });
                        })
                        ->first();

            $exporters_development[$key] = (object)[
                'total_achievement' => (int) $result->total_achievement,
                'total_target'      => (int) $result->total_target,
                'total_achievement_short' => $this->formatForShow($result->total_achievement),
                'total_target_short'      => $this->formatForShow($result->total_target),
                'percentage' => $this->calculateAchievementPercentage($result)
            ];
        }

        return (object)$exporters_development;
    }

    private function formatForShow($num)
    {
        if ($num >= 1000000000) {
            $value = floor($num / 10000000) / 100;
            return ($value == (int)$value ? (int)$value : $value) . ' bil';
        } elseif ($num >= 1000000) {
            $value = floor($num / 10000) / 100;
            return ($value == (int)$value ? (int)$value : $value) . ' juta';
        }
        return ($num == (int)$num ? (int)$num : $num);
    }

    private function calculateAchievementPercentage($result)
    {
        if ($result->total_target <= 0) {
            return 0;
        }
        return (int) round(($result->total_achievement / $result->total_target) * 100);
    }
}